/*************************************************************************
 * The contents of this file are subject to the MYRICOM MYRINET          *
 * EXPRESS (MX) NETWORKING SOFTWARE AND DOCUMENTATION LICENSE (the       *
 * "License"); User may not use this file except in compliance with the  *
 * License.  The full text of the License can found in LICENSE.TXT       *
 *                                                                       *
 * Software distributed under the License is distributed on an "AS IS"   *
 * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied.  See  *
 * the License for the specific language governing rights and            *
 * limitations under the License.                                        *
 *                                                                       *
 * Copyright 2003 - 2004 by Myricom, Inc.  All rights reserved.          *
 *************************************************************************/

/* This file was originally contributed by
 * Brice.Goglin@ens-lyon.org (LIP/INRIA/ENS-Lyon) */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <assert.h>
#include "mx_auto_config.h"
#include "myriexpress.h"
#include "mx_extensions.h"
#if !MX_OS_WINNT
#include <unistd.h>
#include <sys/time.h>
#else
#include "mx_uni.h"
#endif
#include "mx_byteswap.h"
#include "test_common.h"

#define FILTER	0x12345
#define DFLT_EID	1
#define MATCH 0xf0e1d2c3b4a59687ULL
#define CTX (void *)0x26111978
#define CONNECT_DELAY 1000
#define LEN	100

void
usage()
{
  fprintf(stderr, "Usage: mx_iconnect_test [args]\n");
  fprintf(stderr, "-n nic_id - local NIC ID [MX_ANY_NIC]\n");
  fprintf(stderr, "-b board_id - local Board ID [MX_ANY_NIC]\n");
  fprintf(stderr, "-e local_eid - local endpoint ID [%d]\n", DFLT_EID);
  fprintf(stderr, "-r remote_eid - remote endpoint ID [%d]\n", DFLT_EID);
  fprintf(stderr, "-d hostname - destination hostname, required for sender\n");
  fprintf(stderr, "-h - help\n");
}

#if MX_OS_FREEBSD
/* hack around bug in pthreads that gets us into an unkillable state */
#include <signal.h>

void
terminate(int sig)
{

  fprintf(stderr, "Exiting due to signal %d\n", sig);
  _exit(sig == 0 ? 0 : 1);
}
#endif

int main(int argc, char **argv)
{
  mx_return_t rc;
  mx_endpoint_t ep;
  uint64_t nic_id;
  uint16_t my_eid;
  uint64_t his_nic_id;
  uint32_t board_id;
  uint32_t filter;
  uint16_t his_eid;
  mx_endpoint_addr_t his_addr;
  char *rem_host;
  int c;
  extern char *optarg;

#if DEBUG
  extern int mx_debug_mask;
  mx_debug_mask = 0xFFF;
#endif

  /* set up defaults */
  rem_host = NULL;
  filter = FILTER;
  my_eid = DFLT_EID;
  his_eid = DFLT_EID;
  board_id = MX_ANY_NIC;

#if MX_OS_FREEBSD
  (void)signal(SIGINT, terminate);
#endif

  mx_init();
  mx_set_error_handler(MX_ERRORS_RETURN);
  while ((c = getopt(argc, argv, "hd:e:n:b:r:f:w")) != EOF) switch(c) {
  case 'd':
    rem_host = optarg;
    break;
  case 'e':
    my_eid = atoi(optarg);
    break;
  case 'f':
    filter = atoi(optarg);
    break;
  case 'n':
    sscanf(optarg, "%"SCNx64, &nic_id);
    rc = mx_nic_id_to_board_number(nic_id, &board_id);
    if (rc != MX_SUCCESS) {
      fprintf(stderr, "nic_id %012"PRIx64" can't be found\n", nic_id);
      exit(1);
    }
    break;
  case 'b':
    board_id = atoi(optarg);
    break;
  case 'r':
    his_eid = atoi(optarg);
    break;
  case 'h':
  default:
    usage();
    exit(1);
  }

  rc = mx_open_endpoint(board_id, my_eid, filter, NULL, 0, &ep);
  if (rc != MX_SUCCESS) {
    fprintf(stderr, "mx_open_endpoint failed: %s\n", mx_strerror(rc));
    goto abort_with_init;
  }
  
  /* If no host, we are receiver */
  if (rem_host == NULL) {
    mx_request_t recv, send, conn;
    mx_segment_t seg;
    mx_status_t status;
    uint32_t result;
    mx_endpoint_addr_t dest;
    mx_return_t ret;
    uint64_t his_nic_id;
    uint32_t his_eid;
    mx_endpoint_addr_t his_addr;

    printf("Starting mx_iconnect_test receiver\n");

    seg.segment_ptr = malloc(LEN);
    seg.segment_length = LEN;

    ret = mx_irecv(ep, &seg, 1, 0, 0, NULL, &recv);
    assert(ret == MX_SUCCESS);
    ret = mx_wait(ep, &recv, MX_INFINITE, &status, &result);
    assert(ret == MX_SUCCESS);
    printf("wait on irecv result %d status code %d\n", result, status.code);
    assert(status.code == MX_STATUS_SUCCESS);

    memcpy(&dest, &status.source, sizeof(dest));
    ret = mx_decompose_endpoint_addr(dest, &his_nic_id, &his_eid);
    assert(ret == MX_SUCCESS);

    ret = mx_iconnect(ep, his_nic_id, his_eid, filter, MATCH, CTX, &conn);
    assert(ret == MX_SUCCESS);
    ret = mx_wait(ep, &conn, CONNECT_DELAY*2, &status, &result);
    assert(ret == MX_SUCCESS);
    printf("wait on iconnect result %d status code %d\n", result, status.code);
    assert(status.code == MX_STATUS_SUCCESS);
    assert(status.context == CTX);
    assert(status.match_info == MATCH);

    ret = mx_connect(ep, his_nic_id, his_eid, filter, CONNECT_DELAY, &his_addr);
    assert(ret == MX_SUCCESS);
    assert(!memcmp(&status.source, &his_addr, sizeof(his_addr)));

    ret = mx_isend(ep, &seg, 1, dest, 0, NULL, &send);
    assert(ret == MX_SUCCESS);
    ret = mx_wait(ep, &send, MX_INFINITE, &status, &result);
    assert(ret == MX_SUCCESS);
    printf("wait on isend result %d status code %d\n", result, status.code);
    assert(status.code == MX_STATUS_SUCCESS);

    free(seg.segment_ptr);

  } else {
    /* remote hostname implies we are sender */

    mx_request_t recv, send, conn, any;
    mx_segment_t seg;
    mx_status_t status;
    uint32_t result;
    mx_return_t ret;

    printf("Starting mx_iconnect_test send to host %s\n", rem_host);

    /* get address of destination */
    ret = mx_hostname_to_nic_id(rem_host, &his_nic_id);
    if (ret != MX_SUCCESS) {
      fprintf(stderr, "Error getting remote NIC ID: %s\n", mx_strerror(ret));
      goto abort_with_endpoint;
    }

    ret = mx_iconnect(ep, his_nic_id, his_eid, filter, MATCH, CTX, &conn);
    if (ret != MX_SUCCESS) {
      fprintf(stderr, "Error composing remote endpt: %s\n", mx_strerror(ret));
      goto abort_with_endpoint;
    }
    ret = mx_peek(ep, CONNECT_DELAY*2, &any, &result);
    assert(ret == MX_SUCCESS);
    printf("peek result %d\n", result);
    if (result) {
      assert(any == conn);
      ret = mx_test(ep, &conn, &status, &result);
      assert(ret == MX_SUCCESS);
      assert(result);
      printf("test on iconnect result %d status code %d\n", result, status.code);
      assert(status.code == MX_STATUS_SUCCESS);
      assert(status.context == CTX);
      assert(status.match_info == MATCH);
    } else {
      printf("peek got nothing, cancelling iconnect\n");
      ret = mx_cancel(ep, &conn, &result);
      assert(ret == MX_SUCCESS);
      assert(result);
    }

    ret = mx_connect(ep, his_nic_id, his_eid, filter, CONNECT_DELAY, &his_addr);
    if (ret == MX_TIMEOUT) {
      printf("connect timeout\n");
    } else {
      assert(ret == MX_SUCCESS);
      assert(!memcmp(&status.source, &his_addr, sizeof(his_addr)));

      seg.segment_ptr = malloc(LEN);
      seg.segment_length = LEN;

      ret = mx_isend(ep, &seg, 1, his_addr, 0, NULL, &send);
      assert(ret == MX_SUCCESS);

      ret = mx_wait(ep, &send, MX_INFINITE, &status, &result);
      assert(ret == MX_SUCCESS);
      printf("wait on isend result %d status code %d\n", result, status.code);
      assert(status.code == MX_STATUS_SUCCESS);
  
      ret = mx_irecv(ep, &seg, 1, 0, 0, NULL, &recv);
      assert(ret == MX_SUCCESS);

      ret = mx_wait(ep, &recv, MX_INFINITE, &status, &result);
      assert(ret == MX_SUCCESS);
      printf("wait on irecv result %d status code %d\n", result, status.code);
      assert(status.code == MX_STATUS_SUCCESS);

      free(seg.segment_ptr);  
    }
  }

 abort_with_endpoint:
  mx_close_endpoint(ep);
 abort_with_init:
  mx_finalize();

  exit(rc != MX_SUCCESS);
}
